const express = require("express");
const mysql = require("mysql2/promise");
const jwt = require("jsonwebtoken");
const app = express();
const cors = require("cors");
const axios = require("axios");
const fs = require("fs");
const multer = require("multer");
const path = require("path");
const moments = require("moment-timezone");
const moment = require("moment");

app.use(express.json());
app.use(cors());
app.use("/repository", express.static(path.join(__dirname, "repository")));

const dbConfig = {
  host: "localhost",
  user: "root",
  password: "",
  database: "noidmicrofinance",
};

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, "repository");
  },
  filename: (req, file, cb) => {
    cb(null, file.originalname);
  },
});

const upload = multer({ storage });

// API to get repository data
app.get("/api/repository", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM repository");

    if (rows.length > 0) {
      res.json(rows);
    } else {
      res
        .status(404)
        .json({ success: false, errors: "No repository items found" });
    }
  } catch (error) {
    console.error("Error fetching repository data:", error);
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to upload a file and save the URL to the repository table
app.post("/api/repository/upload", upload.single("item"), async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { date } = req.body;
    const file = req.file;

    if (!file) {
      return res
        .status(400)
        .json({ success: false, errors: "No file uploaded" });
    }

    const fileUrl = `/repository/${file.filename}`;

    // Insert the file URL into the repository table
    await connection.execute(
      "INSERT INTO repository (date, file_url) VALUES (?, ?)",
      [date, fileUrl]
    );

    res.json({ success: true, message: "File uploaded successfully", fileUrl });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API for member login
app.post("/login", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM members WHERE name = ? AND phone_number = ?",
      [req.body.username, req.body.phone]
    );

    if (rows.length > 0) {
      const user = rows[0];
      const passCompare = req.body.phone === user.phone_number;

      if (passCompare) {
        const data = {
          user: {
            id: user.id,
          },
        };

        const token = jwt.sign(data, "secret_ecom");
        res.json({ success: true, token });
      } else {
        res.json({ success: false, errors: "Wrong Password" });
      }
    } else {
      res.json({ success: false, errors: "Wrong Username" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API for admin login
app.post("/loginAdmin", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM admins WHERE name = ?",
      [req.body.username]
    );

    if (rows.length > 0) {
      const user = rows[0];
      const passCompare = req.body.password === user.password;

      if (passCompare) {
        const data = {
          user: {
            id: user.id,
          },
        };

        const token = jwt.sign(data, "secret_ecom");
        res.json({ success: true, token });
      } else {
        res.json({ success: false, errors: "Wrong Password" });
      }
    } else {
      res.json({ success: false, errors: "Wrong Username" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Add this endpoint to your existing backend code
app.get("/api/applied-loans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM applied_loans");
    res.json({ success: true, loans: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Add this endpoint to your existing backend code
app.get("/api/taken-loans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM taken_loans");
    res.json({ success: true, loans: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/getMember/:id_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const id_number = req.params.id_number;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM members WHERE id_number = ?",
      [id_number]
    );
    if (rows.length > 0) {
      res.json({ success: true, member: rows[0] });
    } else {
      res.json({ success: false, errors: "Member not found" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API to get all inactive members
app.get("/getInactiveMembers", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM inactive_members");
    res.json({ success: true, members: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API for adding a member
app.post("/addMember", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const {
      registration_number,
      id_number,
      name,
      phone_number,
      acc_number,
      county,
      location,
      email_address,
      next_of_kin_id_number,
      next_of_kin_name,
      relationship,
      next_of_kin_phone_number,
    } = req.body;

    const [result] = await connection.execute(
      "INSERT INTO members (registration_number, id_number, name, phone_number, acc_number, county, location, email_address, next_of_kin_id_number, next_of_kin_name, relationship, next_of_kin_phone_number) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
      [
        registration_number,
        id_number,
        name,
        phone_number,
        acc_number,
        county,
        location,
        email_address,
        next_of_kin_id_number,
        next_of_kin_name,
        relationship,
        next_of_kin_phone_number,
      ]
    );

    res.json({ success: true, message: "Member added successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API to get all members
app.get("/getMembers", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM members");
    res.json({ success: true, members: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API to delete a member and add to inactive members
app.delete("/deleteMember/:id_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const id_number = req.params.id_number;

  try {
    // Start a transaction
    await connection.beginTransaction();

    // Get the member details
    const [memberRows] = await connection.execute(
      "SELECT * FROM members WHERE id_number = ?",
      [id_number]
    );

    if (memberRows.length === 0) {
      return res.json({ success: false, errors: "Member not found" });
    }

    const member = memberRows[0];

    // Insert the member into the inactive_members table
    const [insertResult] = await connection.execute(
      "INSERT INTO inactive_members (registration_number, id_number, name, phone_number, acc_number, county, location, email_address, next_of_kin_id_number, next_of_kin_name, relationship, next_of_kin_phone_number) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
      [
        member.registration_number,
        member.id_number,
        member.name,
        member.phone_number,
        member.acc_number,
        member.county,
        member.location,
        member.email_address,
        member.next_of_kin_id_number,
        member.next_of_kin_name,
        member.relationship,
        member.next_of_kin_phone_number,
      ]
    );

    // Delete the member from the members table
    const [deleteResult] = await connection.execute(
      "DELETE FROM members WHERE id_number = ?",
      [id_number]
    );

    // Commit the transaction
    await connection.commit();

    res.json({ success: true, message: "Member deleted successfully" });
  } catch (error) {
    // Rollback the transaction in case of error
    await connection.rollback();
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Creating API to modify a member
app.put("/modifyMember/:id_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const id_number = req.params.id_number;

  try {
    const {
      registration_number,
      name,
      phone_number,
      acc_number,
      county,
      location,
      email_address,
      next_of_kin_id_number,
      next_of_kin_name,
      relationship,
      next_of_kin_phone_number,
    } = req.body;

    const [result] = await connection.execute(
      "UPDATE members SET registration_number = ?, name = ?, phone_number = ?, acc_number = ?, county = ?, location = ?, email_address = ?, next_of_kin_id_number = ?, next_of_kin_name = ?, relationship = ?, next_of_kin_phone_number = ? WHERE id_number = ?",
      [
        registration_number,
        name,
        phone_number,
        acc_number,
        county,
        location,
        email_address,
        next_of_kin_id_number,
        next_of_kin_name,
        relationship,
        next_of_kin_phone_number,
        id_number,
      ]
    );

    res.json({ success: true, message: "Member updated successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to get all events
app.get("/api/events", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM events ORDER BY date"
    );

    // Add the recurring event for the current month if it doesn't exist
    const currentDate = new Date();
    const recurringEvent = {
      date: `${currentDate.getFullYear()}-${String(
        currentDate.getMonth() + 1
      ).padStart(2, "0")}-12`,
      name: "Contribution Deadline",
      description:
        "This is the contribution deadline that occurs on the 12th of every month.",
      emoji: "📅",
    };

    const existingRecurringEvent = rows.find(
      (event) =>
        event.name === recurringEvent.name &&
        event.emoji === recurringEvent.emoji
    );

    if (!existingRecurringEvent) {
      rows.push(recurringEvent);
    }

    res.json(rows);
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to add a new event
app.post("/api/events/add", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { date, name, description, emoji } = req.body;

    await connection.execute(
      "INSERT INTO events (date, name, description, emoji) VALUES (?, ?, ?, ?)",
      [date, name, description, emoji]
    );

    res.json({ success: true, message: "Event added successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to delete an event
app.delete("/api/events/delete/:id", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { id } = req.params;

    await connection.execute("DELETE FROM events WHERE id = ?", [id]);

    res.json({ success: true, message: "Event deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Get all loan types
app.get("/api/loan-types", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM loan_type");
    res.json({ success: true, loanTypes: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Add a new loan type
app.post("/api/loan-types", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { name, description } = req.body;

    const [result] = await connection.execute(
      "INSERT INTO loan_type (name, description) VALUES (?, ?)",
      [name, description]
    );

    res.json({ success: true, message: "Loan type added successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Delete a loan type
app.delete("/api/loan-types/:id", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const id = req.params.id;

  try {
    const [result] = await connection.execute(
      "DELETE FROM loan_type WHERE id = ?",
      [id]
    );

    res.json({ success: true, message: "Loan type deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/loan-plans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM loan_plan");
    res.json({ success: true, loanPlans: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to add a new loan plan
app.post("/api/loan-plans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { period, interest, penalty } = req.body;

    const [result] = await connection.execute(
      "INSERT INTO loan_plan (period, interest, penalty) VALUES (?, ?, ?)",
      [period, interest, penalty]
    );

    res.json({ success: true, message: "Loan plan added successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to delete a loan plan
app.delete("/api/loan-plans/:id", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const id = req.params.id;

  try {
    const [result] = await connection.execute(
      "DELETE FROM loan_plan WHERE id = ?",
      [id]
    );

    res.json({ success: true, message: "Loan plan deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to get the sum of the amount field in the taken_loans table
app.get("/api/sum-taken-loans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute(
      "SELECT SUM(amount) AS total FROM taken_loans"
    );
    const totalAmount = rows[0].total || 0;
    res.json({ success: true, totalAmount });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to get the sum of the amount field in the applied_loans table
app.get("/api/sum-applied-loans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute(
      "SELECT SUM(amount) AS total FROM applied_loans"
    );
    const totalAmount = rows[0].total || 0;
    res.json({ success: true, totalAmount });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Fetch all fines
app.get("/api/fines", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM fines");
    res.json({ success: true, fines: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Add a new fine
app.post("/api/fines", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { date, person, reason, amount, fined } = req.body;

    const [result] = await connection.execute(
      "INSERT INTO fines (date, person, reason, amount, fined) VALUES (?, ?, ?, ?, ?)",
      [date, person, reason, amount, fined]
    );

    res.json({ success: true, message: "Fine added successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

//api to send sms
app.post("/sendsms", async (req, res) => {
  const { message, mobile } = req.body;

  try {
    const response = await axios.post(
      "https://sms.textsms.co.ke/api/services/sendsms/",
      {
        apikey: "334560520899ec971bfc767c1df7bd94",
        partnerID: "10250",
        message,
        shortcode: "TextSMS",
        mobile,
      }
    );

    res.json(response.data);
  } catch (error) {
    console.error(error);
    res.status(500).send("Error sending SMS");
  }
});

// Add this endpoint to your existing backend code
app.get("/api/member-phone/:name", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const name = req.params.name;

  try {
    const [rows] = await connection.execute(
      "SELECT phone_number FROM members WHERE name = ?",
      [name]
    );
    if (rows.length > 0) {
      res.json({ success: true, phone_number: rows[0].phone_number });
    } else {
      res.json({ success: false, errors: "Member not found" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});
app.get("/api/payments", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM payments");
    res.json({ success: true, payments: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// Add this endpoint to your existing backend code
app.get("/api/sum-payments/:type", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const type = req.params.type;

  try {
    const [rows] = await connection.execute(
      "SELECT SUM(amount) AS total FROM payments WHERE payment = ?",
      [type]
    );
    const totalAmount = rows[0].total || 0;
    res.json({ success: true, totalAmount });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to get taken loans for a specific member
app.get("/api/member-loans/:member_name", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const member_name = req.params.member_name;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM taken_loans WHERE loanee = ?",
      [member_name]
    );
    res.json({ success: true, loans: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.post("/api/apply-loan", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const {
      loanee,
      guarantors,
      amount,
      loanType,
      loanPlanDuration,
      loanPlanInterest,
      loanPlanPenalty,
    } = req.body;

    await connection.execute(
      "INSERT INTO applied_loans (loanee, guarantors, amount, loantype, loanplan_duration, loanplan_interest, loanplan_penalty) VALUES (?, ?, ?, ?, ?, ?, ?)",
      [
        loanee,
        guarantors,
        amount,
        loanType,
        loanPlanDuration,
        loanPlanInterest,
        loanPlanPenalty,
      ]
    );

    res.json({ success: true, message: "Loan applied successfully" });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/loan-plans", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM loan_plan");
    res.json({ success: true, loanPlans: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/loan-types", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const [rows] = await connection.execute("SELECT * FROM loan_type");
    res.json({ success: true, loanTypes: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.post("/api/calculate-loan", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    const { amount, loanPlanId } = req.body;

    const [loanPlanRows] = await connection.execute(
      "SELECT * FROM loan_plan WHERE id = ?",
      [loanPlanId]
    );

    if (loanPlanRows.length === 0) {
      return res
        .status(404)
        .json({ success: false, errors: "Loan plan not found" });
    }

    const loanPlan = loanPlanRows[0];
    const interest = (amount * loanPlan.interest) / 100;
    const penalty = (amount * loanPlan.penalty) / 100;
    const totalPayable = amount + interest;
    const monthlyPayable = totalPayable / loanPlan.period;

    res.json({
      success: true,
      totalPayable,
      monthlyPayable,
      penalty,
    });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.post("/api/apply-loan", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);

  try {
    console.log("Received full request body:", req.body);

    // Destructure with default values
    const {
      loanee = "",
      guarantors = "",
      amount = 0,
      loanType = "",
      loanPlanDuration = 0,
      loanPlanInterest = 0,
      loanPlanPenalty = 0,
    } = req.body;

    console.log("Destructured data:");
    console.log("loanee:", loanee);
    console.log("guarantors:", guarantors);
    console.log("amount:", amount);
    console.log("loanType:", loanType);
    console.log("loanPlanDuration:", loanPlanDuration);
    console.log("loanPlanInterest:", loanPlanInterest);
    console.log("loanPlanPenalty:", loanPlanPenalty);

    // Validate that all required fields are present
    if (
      !loanee ||
      !guarantors ||
      !amount ||
      !loanType ||
      !loanPlanDuration ||
      !loanPlanInterest ||
      !loanPlanPenalty
    ) {
      return res
        .status(400)
        .json({ success: false, errors: "Missing required fields" });
    }

    const [result] = await connection.execute(
      "INSERT INTO applied_loans (loanee, guarantors, amount, loantype, loanplan_duration, loanplan_interest, loanplan_penalty) VALUES (?, ?, ?, ?, ?, ?, ?)",
      [
        loanee,
        guarantors,
        amount,
        loanType,
        loanPlanDuration,
        loanPlanInterest,
        loanPlanPenalty,
      ]
    );

    console.log("Loan application inserted:", result);

    res.json({ success: true, message: "Loan applied successfully" });
  } catch (error) {
    console.error("Error applying loan:", error);
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});
// ACCESS TOKEN FUNCTION - use 'axios'
async function getAccessToken() {
  const consumer_key = "zSAGCj4nsmHGUscubwpmkGkWSC0sGFGMedpdbypoZgzR0MUY"; // REPLACE IT WITH YOUR CONSUMER KEY
  const consumer_secret =
    "zsuInlLK0Sh2Gp1AHFUXNJOKN0A5dUSV84N4PjMCTfvhsyKEYaeSB4qtVNRKYVAC"; // REPLACE IT WITH YOUR CONSUMER SECRET
  const url =
    "https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials";
  const auth =
    "Basic " +
    new Buffer.from(consumer_key + ":" + consumer_secret).toString("base64");

  try {
    const response = await axios.get(url, {
      headers: {
        Authorization: auth,
      },
    });

    const dataresponse = response.data;
    // console.log(data);
    const accessToken = dataresponse.access_token;
    return accessToken;
  } catch (error) {
    throw error;
  }
}

//ACCESS TOKEN ROUTE
app.get("/access_token", (req, res) => {
  getAccessToken()
    .then((accessToken) => {
      res.send("😀 Your access token is " + accessToken);
    })
    .catch(console.log);
});

app.post("/stkpush", (req, res) => {
  const { amount, phoneNumber } = req.body;
  if (!amount) {
    return res.status(400).send("❌ Amount is required");
  }

  getAccessToken()
    .then((accessToken) => {
      const url =
        "https://sandbox.safaricom.co.ke/mpesa/stkpush/v1/processrequest";
      const auth = "Bearer " + accessToken;
      const timestamp = moment().format("YYYYMMDDHHmmss");

      const password = new Buffer.from(
        "174379" +
          "bfb279f9aa9bdbcf158e97dd71a467cd2e0c893059b10f78e6b72ada1ed2c919" +
          timestamp
      ).toString("base64");

      axios
        .post(
          url,
          {
            BusinessShortCode: "174379",
            Password: password,
            Timestamp: timestamp,
            TransactionType: "CustomerPayBillOnline",
            Amount: amount, // Use the amount from the request
            PartyA: phoneNumber, // phone number to receive the stk push
            PartyB: "174379",
            PhoneNumber: phoneNumber,
            CallBackURL: "https://noidhousingbackend.mathenoid.com/stkCallback",
            AccountReference: "mathenoid labs",
            TransactionDesc: "Mpesa Daraja API stk push test",
          },
          {
            headers: {
              Authorization: auth,
            },
          }
        )
        .then((response) => {
          res.send("enter pin to pay");
        })
        .catch((error) => {
          console.log(error);
          res.status(500).send("❌ Request failed");
        });
    })
    .catch(console.log);
});

// API to verify if a registration number belongs to an active member
app.get("/api/verify-member/:registration_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const registration_number = req.params.registration_number;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM members WHERE registration_number = ?",
      [registration_number]
    );
    if (rows.length > 0) {
      res.json({ success: true, member: rows[0] });
    } else {
      res.json({ success: false, errors: "Member not found" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.post("/api/approve-loan/:id", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const loanId = req.params.id;

  try {
    await connection.beginTransaction();

    // Fetch the loan from applied_loans
    const [appliedLoan] = await connection.execute(
      "SELECT * FROM applied_loans WHERE id = ?",
      [loanId]
    );

    if (appliedLoan.length === 0) {
      await connection.rollback();
      return res.status(404).json({ success: false, errors: "Loan not found" });
    }

    const loan = appliedLoan[0];

    // Insert into taken_loans
    await connection.execute(
      "INSERT INTO taken_loans (loanee, loantype, loanplan, date, amount, duedate, comply, guarantors) VALUES (?, ?, ?, NOW(), ?, DATE_ADD(NOW(), INTERVAL ? MONTH), 'ongoing', ?)",
      [
        loan.loanee,
        loan.loantype,
        `${loan.loanplan_duration} months, ${loan.loanplan_interest}% interest, ${loan.loanplan_penalty}% penalty`,
        loan.amount,
        loan.loanplan_duration,
        loan.guarantors.replace(/ /g, "\n"),
      ]
    );

    // Delete from applied_loans
    await connection.execute("DELETE FROM applied_loans WHERE id = ?", [
      loanId,
    ]);

    await connection.commit();
    res.json({ success: true, message: "Loan approved successfully" });
  } catch (error) {
    await connection.rollback();
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.post("/api/reject-loan/:id", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const loanId = req.params.id;

  try {
    await connection.beginTransaction();

    // Fetch the loan from applied_loans
    const [appliedLoan] = await connection.execute(
      "SELECT * FROM applied_loans WHERE id = ?",
      [loanId]
    );

    if (appliedLoan.length === 0) {
      await connection.rollback();
      return res.status(404).json({ success: false, errors: "Loan not found" });
    }

    const loan = appliedLoan[0];

    // Fetch member's phone number
    const [memberRows] = await connection.execute(
      "SELECT phone_number FROM members WHERE name = ?",
      [loan.loanee]
    );

    if (memberRows.length === 0) {
      await connection.rollback();
      return res
        .status(404)
        .json({ success: false, errors: "Member not found" });
    }

    const phoneNumber = memberRows[0].phone_number;

    // Delete from applied_loans
    await connection.execute("DELETE FROM applied_loans WHERE id = ?", [
      loanId,
    ]);

    await connection.commit();

    // Send SMS
    const message = `Dear ${loan.loanee}, we regret to inform you that your loan application of ${loan.amount} has been declined.`;

    try {
      await axios.post("http://localhost:3000/sendsms", {
        message,
        mobile: phoneNumber,
      });
    } catch (smsError) {
      console.error("Error sending SMS:", smsError);
      // Note: We don't return here as the loan has already been rejected
    }

    res.json({ success: true, message: "Loan rejected and SMS sent" });
  } catch (error) {
    await connection.rollback();
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// API to get guarantee requests for a user
app.get("/api/guarantee-requests", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const { registration_number } = req.body;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM applied_loans WHERE guarantors LIKE CONCAT('%\n', ?, '\n%') OR guarantors LIKE CONCAT(?, '\n%') OR guarantors LIKE CONCAT('%\n', ?) OR guarantors = ?",
      [
        registration_number,
        registration_number,
        registration_number,
        registration_number,
      ]
    );

    res.json({ success: true, guaranteeRequests: rows });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/guarantee-requests/:registration_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const registration_number = req.params.registration_number;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM applied_loans WHERE guarantors LIKE CONCAT('%\n', ?, '\n%') OR guarantors LIKE CONCAT(?, '\n%') OR guarantors LIKE CONCAT('%\n', ?) OR guarantors = ?",
      [
        registration_number,
        registration_number,
        registration_number,
        registration_number,
      ]
    );

    const loansWithNames = await Promise.all(
      rows.map(async (loan) => {
        const guarantorNames = await Promise.all(
          loan.guarantors.split(" ").map(async (regNumber) => {
            const [memberRows] = await connection.execute(
              "SELECT name FROM members WHERE registration_number = ?",
              [regNumber]
            );
            return memberRows.length > 0 ? memberRows[0].name : regNumber;
          })
        );

        return { ...loan, guarantors: guarantorNames.join(", ") };
      })
    );

    res.json({ success: true, guaranteeRequests: loansWithNames });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/my-guarantors/:registration_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const registration_number = req.params.registration_number;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM taken_loans WHERE loanee LIKE CONCAT('%\n', ?, '\n%') OR guarantors LIKE CONCAT(?, '\n%') OR guarantors LIKE CONCAT('%\n', ?) OR guarantors = ?",
      [
        registration_number,
        registration_number,
        registration_number,
        registration_number,
      ]
    );

    const loansWithNames = await Promise.all(
      rows.map(async (loan) => {
        const guarantorNames = await Promise.all(
          loan.guarantors.split(" ").map(async (regNumber) => {
            const [memberRows] = await connection.execute(
              "SELECT name FROM members WHERE registration_number = ?",
              [regNumber]
            );
            return memberRows.length > 0 ? memberRows[0].name : regNumber;
          })
        );

        return { ...loan, guarantors: guarantorNames.join(", ") };
      })
    );

    res.json({ success: true, myGuarantors: loansWithNames });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.get("/api/my-guarantees/:registration_number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const registration_number = req.params.registration_number;

  try {
    const [rows] = await connection.execute(
      "SELECT * FROM taken_loans WHERE guarantors LIKE CONCAT('%\n', ?, '\n%') OR guarantors LIKE CONCAT(?, '\n%') OR guarantors LIKE CONCAT('%\n', ?) OR guarantors = ?",
      [
        registration_number,
        registration_number,
        registration_number,
        registration_number,
      ]
    );

    const loansWithNames = await Promise.all(
      rows.map(async (loan) => {
        const guarantorNames = await Promise.all(
          loan.guarantors.split(" ").map(async (regNumber) => {
            const [memberRows] = await connection.execute(
              "SELECT name FROM members WHERE registration_number = ?",
              [regNumber]
            );
            return memberRows.length > 0 ? memberRows[0].name : regNumber;
          })
        );

        return { ...loan, guarantors: guarantorNames.join(", ") };
      })
    );

    res.json({ success: true, myGuarantees: loansWithNames });
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

// // API to get my guarantors
// app.get("/api/my-guarantors/:registration_number", async (req, res) => {
//   const connection = await mysql.createConnection(dbConfig);
//   const registration_number = req.params.registration_number;

//   try {
//     const [rows] = await connection.execute(
//       "SELECT * FROM taken_loans WHERE loanee = ?",
//       [registration_number]
//     );

//     const loansWithNames = await Promise.all(
//       rows.map(async (loan) => {
//         const guarantorNames = await Promise.all(
//           loan.guarantors.split(" ").map(async (regNumber) => {
//             const [memberRows] = await connection.execute(
//               "SELECT name FROM members WHERE registration_number = ?",
//               [regNumber]
//             );
//             return memberRows.length > 0 ? memberRows[0].name : regNumber;
//           })
//         );

//         return { ...loan, guarantors: guarantorNames.join(", ") };
//       })
//     );

//     res.json({ success: true, myGuarantors: loansWithNames });
//   } catch (error) {
//     res.status(500).json({ success: false, errors: error.message });
//   } finally {
//     await connection.end();
//   }
// });
// // API to get my guarantees
// app.get("/api/my-guarantees/:registration_number", async (req, res) => {
//   const connection = await mysql.createConnection(dbConfig);
//   const registration_number = req.params.registration_number;

//   try {
//     const [rows] = await connection.execute(
//       "SELECT * FROM taken_loans WHERE FIND_IN_SET(?, guarantors) > 0",
//       [registration_number]
//     );

//     const loansWithNames = await Promise.all(
//       rows.map(async (loan) => {
//         const guarantorNames = await Promise.all(
//           loan.guarantors.split(" ").map(async (regNumber) => {
//             const [memberRows] = await connection.execute(
//               "SELECT name FROM members WHERE registration_number = ?",
//               [regNumber]
//             );
//             return memberRows.length > 0 ? memberRows[0].name : regNumber;
//           })
//         );

//         return { ...loan, guarantors: guarantorNames.join(", ") };
//       })
//     );

//     res.json({ success: true, myGuarantees: loansWithNames });
//   } catch (error) {
//     res.status(500).json({ success: false, errors: error.message });
//   } finally {
//     await connection.end();
//   }
// });

// // API to get member name by registration number
// app.get("/api/member-name/:registration_number", async (req, res) => {
//   const connection = await mysql.createConnection(dbConfig);
//   const registration_number = req.params.registration_number;

//   try {
//     const [rows] = await connection.execute(
//       "SELECT name FROM members WHERE registration_number = ?",
//       [registration_number]
//     );
//     if (rows.length > 0) {
//       res.json({ success: true, name: rows[0].name });
//     } else {
//       res.json({ success: false, errors: "Member not found" });
//     }
//   } catch (error) {
//     res.status(500).json({ success: false, errors: error.message });
//   } finally {
//     await connection.end();
//   }
// });

// API to get registration number by user name
app.post("/api/registration-number", async (req, res) => {
  const connection = await mysql.createConnection(dbConfig);
  const { name } = req.body;

  if (!name) {
    return res.status(400).json({ success: false, errors: "Name is required" });
  }

  try {
    const [rows] = await connection.execute(
      "SELECT registration_number FROM members WHERE name = ?",
      [name]
    );
    if (rows.length > 0) {
      res.json({
        success: true,
        registration_number: rows[0].registration_number,
      });
    } else {
      res.json({ success: false, errors: "Member not found" });
    }
  } catch (error) {
    res.status(500).json({ success: false, errors: error.message });
  } finally {
    await connection.end();
  }
});

app.listen(3000, () => {
  console.log("Server is running on port 3000");
});
